/*
 Copyright (C) 2017 Apple Inc. All Rights Reserved.
 See LICENSE.txt for this sample’s licensing information
*/

#import "NowPlayingLogger.h"
#import <MediaPlayer/MediaPlayer.h>

/*
 * NowPlayingLogger is a drop-in class that can be used to log changes to the Now Playing Info, 
 * which is useful when debugging and implementing integrations with Universal Search / TV app.
 * 
 * NOTE: ensure that you ONLY include and call methods from this class in DEBUG builds. Do not
 *       include it and use it in production applications.
 *
 * Usage: 
 *       1. drop the NowPlayingLogger.m/NowPlayingLogger.h files into your project
 *
 *       2. Retain a strong reference to the NowPlayingLogger instance as a property
 *          of your application delegate:
 *
 *            @property (nonatomic, strong) NowPlayingLogger *nowPlayingLogger;
 *
 *       3. In application:didFinishLaunchingWithOptions:, initialize the NowPlayingLogger:
 *
 *            [self setNowPlayingLogger:[[NowPlayingLogger alloc] init]];
 *
 *       4. In applicationDidBecomeActive, call start:
 *
 *            [[self nowPlayingLogger] start];
 *
 *       5. In applicationDidEnterBackground, call stop:
 *
 *          [[self nowPlayingLogger] stop];
 *
 *       6. When running and debugging the application, the Now Playing Info will be 
 *          logged to the console when changes occur, prepended with "[NowPlayingLogger]"
 */

@interface NowPlayingLogger ()

@property (nonatomic, strong) NSThread *thread;
@property (nonatomic, copy) NSDictionary *lastNowPlayingInfo;

@end

@implementation NowPlayingLogger

- (instancetype)init
{
#ifdef DEBUG
    [self setThread:[[NSThread alloc] initWithTarget:self selector:@selector(logger) object:nil]];
#endif
    return [super init];
}

/* start starts logging changes to the Now Playing Info, by starting the logger thread */
- (void)start
{
#ifdef DEBUG
    [[self thread] start];
#endif
}

/* stop marks the logging thread as cancelled, which is read by the logger thread to know when to quit running */
- (void)stop
{
#ifdef DEBUG
    [[self thread] cancel];
#endif
}

/* logger is the main function for the thread that logs changes to the Now playing Info. 
 * Every 100us, it pulls the NowPlayingInfo, and compares it to the Now Playing Info
 * that was retrieved in the last iteration. If it's different, it logs it to the console
 */
-(void)logger
{
#ifdef DEBUG
    /* while the thread is not marked as cancelled, check to see if the current Now Playing
     * Info has changed since the last iteration, and log it if so */
    while (![[self thread] isCancelled])
    {
        NSDictionary *npi = [[MPNowPlayingInfoCenter defaultCenter] nowPlayingInfo];
        
        if (npi != [self lastNowPlayingInfo])
        {
            [self setLastNowPlayingInfo:npi];
            NSLog(@"[NowPlayingLogger] %@", npi);

            if (![self isRegisteredForRemoteCommands])
            {
                NSLog(@"[NowPlayingLogger] ERROR: You are not registered for remote commands, which is required. See MPRemoteCommandCenter.");
            }
        }
        
        usleep(100);
    }
#endif
}

- (BOOL)isRegisteredForRemoteCommands
{
    @try {
        NSArray *activeCommands = [[MPRemoteCommandCenter sharedCommandCenter] valueForKeyPath:@"activeCommands.hasTargets"];
        return ([activeCommands count] && [activeCommands indexOfObject:@YES] != NSNotFound);
    }
    @catch (NSException *exception) {
        NSLog(@"warning: failed to determine if registered for remote commands");
    }

    return NO;
}

@end
